/*
  H Y P E R M A M M U T
 
  Copyright (C) 2006 Nasca Octavian Paul
  Author: Nasca Octavian Paul

  This program is free software; you can redistribute it and/or modify
  it under the terms of version 2 of the GNU General Public License 
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License (version 2) for more details.

  You should have received a copy of the GNU General Public License (version 2)
  along with this program; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
*/

#include "FilterS.h"
#include <math.h>
using namespace std;

FilterS::FilterS():Process("SNDFREQ_FILTER","Filter"){
    //add inputs/outputs
    set_n_inputs(1);
    set_n_outputs(1);
    
    //add parameters
    add_par_choice("TYPE","Type","LowPass|HighPass|BandPass|BandStop|Comb|PADFilter","LowPass");

    add_par_real("FREQ1","Frequency (Hz)",0.01,22050,1000,LINEAR,20.0);
    add_par_real("FREQ2","Frequency (Hz)",0.01,22050,2000,LINEAR,20.0);
    add_par_real("SMOOTH","Smooth (Hz)",0.01,10000,1,EXP,-10.0);

    add_par_real("STRENGTH","Strength",0.5,50.0,1.0,EXP,0.0);
    add_par_real("PHASE","Phase",-PI,PI,0.0,LINEAR,0.0);
    add_par_real("STRETCH","Stretch",-2.0,2.0,0.0,LINEAR,0.0);

    add_par_real("BANDWIDTH","Bandwidth (cents)",0.1,250.0,30.0,EXP,-10.0);
    add_par_int("NHARMONICS","Number of harmonics(overtones)",1,256,40,20.0);
    add_par_bool("GAUSS","Gauss mode",true);

    update_parameters("");
};

Object::Type FilterS::get_input_type(int n){
    switch(n){
	case 0: return Object::SNDFREQ;
    };
    return Object::NONE;
};
Object::Type FilterS::get_output_type(int n){
    switch(n){
	case 0: return Object::SNDFREQ;
    };
    return Object::NONE;
};

bool FilterS::update_parameters(string source_par){
    string type=get_par_choice("TYPE");

    bool passfilters=(type.find("Pass")!=-1)||(type.find("Band")!=-1);
    set_enabled_par("FREQ1",true);
    set_enabled_par("FREQ2",(type.find("Band")!=-1)&&passfilters);
    set_enabled_par("SMOOTH",passfilters);

    bool combfilters=(type.find("Comb")!=-1);
    set_enabled_par("STRENGTH",combfilters);
    set_enabled_par("PHASE",combfilters);
    set_enabled_par("STRETCH",combfilters);
    
    bool bwfilters=(type.find("PADFilter")!=-1);
    set_enabled_par("BANDWIDTH",bwfilters);
    set_enabled_par("NHARMONICS",bwfilters);
    set_enabled_par("GAUSS",bwfilters);

    if (source_par=="TYPE") return true;
	else return false;
};


bool FilterS::do_process(){
    string type=get_par_choice("TYPE");
    
    SndFreq *input=dynamic_cast<SndFreq *>(inputs[0]);
    SndFreq *output=dynamic_cast<SndFreq *>(outputs[0]);
    output->copyinfofrom(input);


    REALTYPE freq1=get_par_real("FREQ1");
    REALTYPE freq2=get_par_real("FREQ2");
    REALTYPE smooth=get_par_real("SMOOTH");
    REALTYPE strength=get_par_real("STRENGTH");
    REALTYPE phase=get_par_real("PHASE");
    REALTYPE stretch=get_par_real("STRETCH");
    REALTYPE bandwidth=get_par_real("BANDWIDTH");
    int nharmonics=get_par_int("NHARMONICS");
    bool gauss=get_par_bool("GAUSS");
    
    if (type=="LowPass"){
	lpf(input,output,freq1,smooth);
    };
    if (type=="HighPass"){
	hpf(input,output,freq1,smooth);
    };
    if (type=="BandPass"){
	bpf(input,output,freq1,freq2,smooth);
    };
    if (type=="BandStop"){
	bsf(input,output,freq1,freq2,smooth);
    };
    if (type=="Comb"){
	combf(input,output,freq1,strength,phase,stretch);
    };
    if (type=="PADFilter"){
	bandwidthf(input,output,freq1,bandwidth,nharmonics,gauss);
    };

    return true;
};


void FilterS::lpf(SndFreq *sndfreq1,SndFreq *sndfreq2,REALTYPE freq,REALTYPE smooth){
    filter(sndfreq1,sndfreq2,false,-1,freq,smooth);
};

void FilterS::hpf(SndFreq *sndfreq1,SndFreq *sndfreq2,REALTYPE freq,REALTYPE smooth){
    filter(sndfreq1,sndfreq2,true,-1,freq,smooth);
};

void FilterS::bpf(SndFreq *sndfreq1,SndFreq *sndfreq2,REALTYPE freq1,REALTYPE freq2,REALTYPE smooth){
    if (freq2<freq1){
	REALTYPE tmp=freq1;
	freq1=freq2;
	freq2=tmp;
    };
    filter(sndfreq1,sndfreq2,false,freq1,freq2,smooth);
};

void FilterS::bsf(SndFreq *sndfreq1,SndFreq *sndfreq2,REALTYPE freq1,REALTYPE freq2,REALTYPE smooth){
    if (freq2<freq1){
	REALTYPE tmp=freq1;
	freq1=freq2;
	freq2=tmp;
    };
    filter(sndfreq1,sndfreq2,true,freq1,freq2,smooth);
};


void FilterS::filter(SndFreq *sndfreq1,SndFreq *sndfreq2,bool invertamp,REALTYPE freq1,REALTYPE freq2,REALTYPE smooth){
    int nx=sndfreq1->getnx();
    if (smooth<1e-3) smooth=1e-3;
    if (sndfreq1!=sndfreq2) sndfreq2->reset(nx);
    if (nx==0) return;
    
    REALTYPE inx=sndfreq1->info_get_samplerate()*1.0/nx;
    for (int i=0;i<nx;i++){
	REALTYPE c,s,amp1=1.0,amp2=1.0;
	REALTYPE f=i*inx;
	sndfreq1->get(i,c,s);

	//hpf
	if (freq1>0){
	    if (f>freq1+smooth*0.5){
		amp1=0.0;
	    }else{
		if (f>freq1-smooth*0.5){
		    REALTYPE f1=(f-(freq1-smooth*0.5))/smooth;
		    amp1=(cos(f1*PI)+1.0)*0.5;
		};
	    };
	    amp1=1.0-amp1;
	};

	//lpf
	if (freq2>0){
	    if (f>freq2+smooth*0.5){
		amp2=0.0;
	    }else{
		if (f>freq2-smooth*0.5){
		    REALTYPE f1=(f-(freq2-smooth*0.5))/smooth;
		    amp2=(cos(f1*PI)+1.0)*0.5;
		};
	    };
	};


	REALTYPE amp=amp1*amp2;
	if (invertamp) amp=1.0-amp;
	sndfreq2->set(i,c*amp,s*amp);
    };

};


void FilterS::combf(SndFreq *sndfreq1,SndFreq *sndfreq2,REALTYPE freq,REALTYPE strength,REALTYPE phase,REALTYPE stretch){
    int nx=sndfreq1->getnx();
    if (freq<1e-3) freq=1e-3;
    if (sndfreq1!=sndfreq2) sndfreq2->reset(nx);
    if (nx==0) return;

    stretch=pow(2.0,-stretch);
    
    REALTYPE samplerate=sndfreq1->info_get_samplerate();
    for (int i=0;i<nx;i++){
	REALTYPE c,s,amp=0.0;
	REALTYPE x=i/(REALTYPE) nx;// de la 0 la 1
	REALTYPE f=pow(x*10.0,stretch)*samplerate*0.5*0.1;
	sndfreq1->get(i,c,s);

	amp=fabs(cos(f*PI/freq*2.0+phase));
	amp=pow(amp,strength);
	sndfreq2->set(i,c*amp,s*amp);
    };

};


void FilterS::bandwidthf(SndFreq *sndfreq1,SndFreq *sndfreq2,REALTYPE freq,REALTYPE bandwidth,int nharmonics,bool gauss){
    int nx=sndfreq1->getnx();
    if (freq<10.0) freq=10.0;
    if (sndfreq1!=sndfreq2) sndfreq2->reset(nx);
    if (nx==0) return;

    REALTYPE *amp=new REALTYPE[nx];
    for (int i=0;i<nx;i++) amp[i]=0.0;

    int samplerate=sndfreq1->info_get_samplerate();  
    for (int nh=1;nh<=nharmonics;nh++){//for each harmonic
	REALTYPE bw_Hz;//bandwidth of the current harmonic measured in Hz
        REALTYPE bwi;
	REALTYPE fi;
	REALTYPE f=nh*freq;
	
	if (f>=nx/2) break;

        bw_Hz=(pow(2.0,bandwidth/1200.0)-1.0)*f;
	bwi=bw_Hz/(2.0*samplerate);
	fi=f/samplerate;

	REALTYPE sum=0.0;
	REALTYPE max=0.0;
	for (int i=1;i<nx/2;i++){//todo: optimize here
	    REALTYPE hprofile;
	    hprofile=profile((i/(REALTYPE)nx)-fi,bwi);
	    amp[i]+=hprofile;
	    if (max<hprofile) max=hprofile;
	    sum+=hprofile;
	};
    };
    
    REALTYPE max=0.0;
    for (int i=1;i<nx/2;i++){
	if (amp[i]>max) max=amp[i];
    };
    if (max<1e-8) max=1e-8;


    for (int i=1;i<nx/2;i++){
	REALTYPE c,s;
	sndfreq1->get(i,c,s);
	REALTYPE a=amp[i]/max;
	if (!gauss) a=(a<0.368?0.0:1.0);
	sndfreq2->set(i,c*a,s*a);
    };
    
    
    delete(amp);

};

REALTYPE FilterS::profile(REALTYPE fi, REALTYPE bwi){
    REALTYPE x=fi/bwi;
    x*=x;
    if (x>14.71280603) return 0.0;
    return exp(-x);///bwi;
};
