/*
  H Y P E R M A M M U T
 
  Copyright (C) 2006 Nasca Octavian Paul
  Author: Nasca Octavian Paul

  This program is free software; you can redistribute it and/or modify
  it under the terms of version 2 of the GNU General Public License 
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License (version 2) for more details.

  You should have received a copy of the GNU General Public License (version 2)
  along with this program; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
*/

#include "PhaseChangeS.h"
#include "../../Generator/Random.h"
#include <math.h>
using namespace std;

PhaseChangeS::PhaseChangeS():Process("SNDFREQ_PHASECHANGE","Phase Change"){
    //add inputs/outputs
    set_n_inputs(1);
    set_n_outputs(1);
    
    //add parameters
    add_par_choice("TYPE","Type","Phase|GroupDelay","Phase");
    add_par_choice("TYPEPHASE","Change Type","Add|Multiply|Randomize|Amplitude|Reverse","Add");
    add_par_choice("TYPEGROUPDELAY","Change Type","Add|Linear|Randomize|Amplitude|NonLinearSquare|NonLinearSine","Linear");

    //la phase (in functie de typephase)
    add_par_real("PHAMOUNTADD","Amount",-PI,PI,0.0,LINEAR,0.0);
    add_par_real("PHAMOUNTMUL","Amount",0.0,10.0,1.0,LINEAR,15.0);
    add_par_real("PHAMOUNTRAND","Amount",0.0,1.0,0.1,LINEAR,10.0);
    add_par_real("PHAMOUNTAMP","Amount",0.0,100.0,1.0,LINEAR,30.0);

    //la groupdelay
    add_par_real("GDAMOUNTADD","Amount",-PI,PI,0.0,LINEAR,0.0);
    add_par_real("GDAMOUNTRAND","Amount",0.0,100.0,1.0,LINEAR,15.0);
    add_par_real("GDAMOUNTRANDFREQ","Frequency",0.1,100.0,1.0,EXP,0.0);
    add_par_real("GDAMOUNTAMP","Amount",0.0,5.0,0.1,LINEAR,40.0);
    add_par_real("GDAMOUNTLIN","Amount",0.0,10.0,0.1,LINEAR,20.0);
    add_par_real("GDAMOUNTNONLINSQR","Amount",0.0,10.0,0.1,LINEAR,20.0);
    add_par_real("GDAMOUNTNONLINSIN","Amount",0.0,10.0,0.1,LINEAR,20.0);
    add_par_bool("RELATIVE","Relative Amount",false);

    add_par_choice("GDMODEAMOUNT","Amount Mode","Positive|Negative","Positive");
    
    update_parameters("");
};

Object::Type PhaseChangeS::get_input_type(int n){
    switch(n){
	case 0: return Object::SNDFREQ;
    };
    return Object::NONE;
};
Object::Type PhaseChangeS::get_output_type(int n){
    switch(n){
	case 0: return Object::SNDFREQ;
    };
    return Object::NONE;
};
bool PhaseChangeS::update_parameters(string source_par){
    bool phase_enabled=(get_par_choice("TYPE")=="Phase");
    bool groupdelay_enabled=(get_par_choice("TYPE")=="GroupDelay");
    string phase_type_str=get_par_choice("TYPEPHASE");
    string groupdelay_type_str=get_par_choice("TYPEGROUPDELAY");

    set_enabled_par("TYPEPHASE",phase_enabled);
    set_enabled_par("PHAMOUNTADD",(phase_type_str=="Add")&&phase_enabled);
    set_enabled_par("PHAMOUNTMUL",(phase_type_str=="Multiply")&&phase_enabled);
    set_enabled_par("PHAMOUNTRAND",(phase_type_str=="Randomize")&&phase_enabled);
    set_enabled_par("PHAMOUNTAMP",(phase_type_str=="Amplitude")&&phase_enabled);

    set_enabled_par("TYPEGROUPDELAY",groupdelay_enabled);
    set_enabled_par("GDAMOUNTADD",(groupdelay_type_str=="Add")&&groupdelay_enabled);
    set_enabled_par("GDAMOUNTRAND",(groupdelay_type_str=="Randomize")&&groupdelay_enabled);
    set_enabled_par("GDAMOUNTRANDFREQ",(groupdelay_type_str=="Randomize")&&groupdelay_enabled);
    set_enabled_par("GDAMOUNTAMP",(groupdelay_type_str=="Amplitude")&&groupdelay_enabled);
    set_enabled_par("GDAMOUNTLIN",(groupdelay_type_str=="Linear")&&groupdelay_enabled);
    set_enabled_par("GDAMOUNTNONLINSQR",(groupdelay_type_str=="NonLinearSquare")&&groupdelay_enabled);
    set_enabled_par("GDAMOUNTNONLINSIN",(groupdelay_type_str=="NonLinearSine")&&groupdelay_enabled);
    set_enabled_par("RELATIVE",groupdelay_enabled);

    bool groupdelay_mode_amount=((groupdelay_type_str=="Amplitude")||
	(groupdelay_type_str=="Linear")||(groupdelay_type_str=="NonLinearSine")||(groupdelay_type_str=="NonLinearSquare"));
    set_enabled_par("GDMODEAMOUNT",groupdelay_enabled&&groupdelay_mode_amount);

    if ((source_par=="TYPE")||(source_par=="TYPEPHASE")||(source_par=="TYPEGROUPDELAY")) return true;
	else return false;

};
bool PhaseChangeS::do_process(){
    SndFreq *input=dynamic_cast<SndFreq *>(inputs[0]);
    SndFreq *output=dynamic_cast<SndFreq *>(outputs[0]);
    output->copyinfofrom(input);

    string type=get_par_choice("TYPE");
    if (type=="Phase") {
	string phase_type_str=get_par_choice("TYPEPHASE");
	PhaseChangeType phase_type;
	REALTYPE amount=1.0;

	if (phase_type_str=="Add") {
	    phase_type=PHADD;
	    amount=get_par_real("PHAMOUNTADD");
	};
	if (phase_type_str=="Multiply") {
	    phase_type=PHMUL;
	    amount=get_par_real("PHAMOUNTMUL");
	};
	if (phase_type_str=="Randomize") {
	phase_type=PHRAND;
	amount=get_par_real("PHAMOUNTRAND");
	};
	if (phase_type_str=="Amplitude") {
	    phase_type=PHAMP;
	    amount=get_par_real("PHAMOUNTAMP");
	};
	if (phase_type_str=="Reverse") {
	    phase_type=PHREV;
	};
	change_phase(input,output,phase_type,amount);
    };

    if (type=="GroupDelay") {
	string groupdelay_type_str=get_par_choice("TYPEGROUPDELAY");
	bool relative=get_par_bool("RELATIVE");
	
	GroupDelayChangeType groupdelay_type;
	REALTYPE amount=1.0;
	REALTYPE freq=0.1;
	
	if (groupdelay_type_str=="Add") {
	    groupdelay_type=GDADD;
	    amount=get_par_real("GDAMOUNTADD");
	};

	if (groupdelay_type_str=="Linear") {
	    groupdelay_type=GDLIN;
	    amount=get_par_real("GDAMOUNTLIN");
	    if (get_par_choice("GDMODEAMOUNT")=="Negative") amount=-amount;
	};

	if (groupdelay_type_str=="NonLinearSquare") {
	    groupdelay_type=GDNONLINSQR;
	    amount=get_par_real("GDAMOUNTNONLINSQR");
	    if (get_par_choice("GDMODEAMOUNT")=="Negative") amount=-amount;
	};
	if (groupdelay_type_str=="NonLinearSine") {
	    groupdelay_type=GDNONLINSIN;
	    amount=get_par_real("GDAMOUNTNONLINSIN");
	    if (get_par_choice("GDMODEAMOUNT")=="Negative") amount=-amount;
	};


	if (groupdelay_type_str=="Randomize") {
	    groupdelay_type=GDRAND;
	    amount=get_par_real("GDAMOUNTRAND")*0.01;
	    freq=get_par_real("GDAMOUNTRANDFREQ")*0.01;
	};

	if (groupdelay_type_str=="Amplitude") {
	    groupdelay_type=GDAMP;
	    amount=get_par_real("GDAMOUNTAMP");
	    amount=amount*amount;
	    if (get_par_choice("GDMODEAMOUNT")=="Negative") amount=-amount;
	};

	if (!relative) {
	    REALTYPE seconds=input->getnx()/input->info_get_samplerate();
	    REALTYPE rel=seconds/60.0;
	    amount/=rel;
	};
	change_groupdelay(input,output,groupdelay_type,amount,freq);
    };


    return true;
};

void PhaseChangeS::change_phase(SndFreq *sndfreq1,SndFreq *sndfreq2,PhaseChangeType type,REALTYPE amount){
    Random rand;
    int nx=sndfreq1->getnx();
    if (sndfreq1!=sndfreq2) sndfreq2->reset(nx);
    REALTYPE max_amplitude=sndfreq1->get_max_amplitude();
    
    for (int i=0;i<nx/2;i++){
	REALTYPE c,s;
	sndfreq1->get(i,c,s);

	REALTYPE phase=atan2(s,c);
	REALTYPE amp=sqrt(s*s+c*c);
	switch (type){
	    case PHADD:
		phase+=amount;		
	    break;
	    case PHMUL:
		phase*=amount;
	    break;
	    case PHRAND:
		phase+=(rand.get()-0.5)*2.0*amount*2.0*PI;
	    break;
	    case PHAMP:
		phase+=amp/max_amplitude*amount*2.0*PI;
	    break;
	    case PHREV:
		phase=-phase;
	    break;
	};

	c=amp*cos(phase);
	s=amp*sin(phase);
	sndfreq2->set(i,c,s);
    };
};



void PhaseChangeS::change_groupdelay(SndFreq *sndfreq1,SndFreq *sndfreq2,GroupDelayChangeType type,REALTYPE amount,REALTYPE freq){
    Random rand;
    freq*=freq;
    int nx=sndfreq1->getnx();
    if (sndfreq1!=sndfreq2) sndfreq2->reset(nx);
    if (nx==0) return;
    REALTYPE max_amplitude=sndfreq1->get_max_amplitude();
    REALTYPE iphase=0.0,ophase=0.0,groupdelay=0.0;
    
    REALTYPE inv_nx2=2.0/nx;
    
    REALTYPE randf1=rand.get1()*freq;
    REALTYPE randf2=rand.get1()*freq;
    
    REALTYPE coef=1.0/sqrt(freq/16.0);
    
    for (int i=0;i<nx/2;i++){
	REALTYPE c,s;
	sndfreq1->get(i,c,s);

	REALTYPE phase=atan2(s,c);
	REALTYPE amp=sqrt(s*s+c*c);

	groupdelay=(phase-iphase);
	iphase=phase;

	switch (type){
	    case GDADD:
		groupdelay+=amount;
	    break;
	    case GDRAND:
		randf2=randf2*(1.0-freq)+rand.get1()*freq;
		randf1=randf1*(1.0-freq)+randf2*freq;
		groupdelay+=randf1*amount*2.0*PI*coef;
	    break;
	    case GDAMP:
		groupdelay+=amp/max_amplitude*amount*2.0*PI;
	    break;
	    case GDLIN:
		groupdelay+=i*inv_nx2*amount*2.0*PI;
	    break;
	    case GDNONLINSQR:
		groupdelay+=amount*groupdelay*groupdelay/(4.0*PI);
	    break;
	    case GDNONLINSIN:
		groupdelay+=amount*sin(groupdelay);
	    break;
	};

	ophase+=groupdelay;

	phase=ophase;
	
	c=amp*cos(phase);
	s=amp*sin(phase);
	sndfreq2->set(i,c,s);
    };
};


